--
-- GM HC11 CPU Core
-- Copyright (C) Green Mountain Computing Systems, 2000
-- All rights reserved.
--
-- This file may not be freely distributed.  This file has been provided
-- under the terms of the GM Core License Agreement in license.txt.
--
-- srec_rom.vhd : This is a VHDL behavioral implementation of a ROM loaded
--          from a file.  This file uses the S-record format generated by
--          the HC11 assembler.  Unlike the real thing, this ROM will accept
--          CPU writes to alter the memory.  In order to simulate the execution
--          of a program written for the HC11, it should be assembled and
--          loaded in the ROM.  Be sure to include a value for the reset
--          vector that the CPU uses to jump to the start of the program (see
--          the example hexout.asm).
--
-- 8/15/00 : Created - Scott Thibault
--

library IEEE;
use ieee.std_logic_1164.all;
use ieee.numeric_std.all;

entity srec_rom is
  generic (filename : string := "rom.s19";
           base : natural :=16#D000#;    -- Location of EPROM in memory map
           size : natural :=16#3000#);   -- Size of EPROM
  port (E : in std_logic;
        ph1, ph2, reset : in std_logic;

        as, bus_rw : in std_logic;
        bus_addr : in std_logic_vector (15 downto 0);
        bus_data : inout std_logic_vector (7 downto 0));
end srec_rom;

use textio.all;

architecture behavior of srec_rom is
begin
  do_rom: process
    -----------------------
    -- Function a2i
    --   This function converts a hexadecimal character into its integer
    --   value.
    -----------------------
    function a2i(ch: in character) return integer is
      variable val,ret: integer;
    begin
      val:=character'pos(ch);
      if ((val>=48) and (val<58)) then
        ret:=val-48;        -- ch between '0' and '9'
      elsif ((val>=65) and (val<=70)) then
        ret:=val-55;    -- ch between 'A' and 'F'
      elsif ((val>=97) and (val<=102)) then
        ret:=val-87;    -- ch between 'a' and 'f'
      else
        ret:=0;
        assert FALSE report "Invalid hexidecimal digit "&ch severity warning;
      end if;
      return(ret);
    end a2i;

    -- variables for reading S-record file 
    file rom_file: text open read_mode is filename;
    variable srec,str : line;
    variable done : boolean;
    variable ch,hi,lo : character;
    variable val,num,address,n_recs,total : integer;

    -- variables for reads and writes
    type mem_array is array (0 to size-1) of std_logic_vector(7 downto 0);
    variable mem : mem_array;
    variable rw : std_logic;
  begin
    n_recs:=0;       -- Count records for stastics
    total:=0;        -- Total bytes for stastics

    write(str,"loading eprom.s19...");
    writeline(output,str);

    done:=FALSE;
    while (not done) loop
      n_recs:=n_recs+1;
      readline(rom_file,srec);     -- Read next record from file

      -- First character must be 'S'
      read(srec,ch);
      assert ch='S' report "Malformed S-record file (eprom.s19)" 
        severity failure;

      -- Next character is the type of S-record '1'-data '9'-last S-record.
      -- All other types are not supported.
      read(srec,ch);
      case ch is
        -- Data record
        when '1' =>
          -- read record length
          read(srec,hi);
          read(srec,lo);
          num:=a2i(hi)*16+a2i(lo);
          -- read address high byte
          read(srec,hi);
          read(srec,lo);
          address:=a2i(hi)*16+a2i(lo);
          -- read address low byte
          read(srec,hi);
          read(srec,lo);
          address:=address*256+a2i(hi)*16+a2i(lo);
          -- read data
          total:=total+num-3;  -- num includes two address bytes and the chksum
          address:=address-base; -- adjust for indexing
          for i in 1 to num-3 loop
            read(srec,hi);
            read(srec,lo);
            val:=a2i(hi)*16+a2i(lo);
            if ((address>=0) and (address<size)) then
              mem(address):=std_logic_vector(to_unsigned(val,8));
            else 
              assert FALSE report "non-addressable data in eprom.s19"
                 severity warning;
            end if;
            address:=address+1;
          end loop;

        -- End of S-records
        when '9' =>
          done:=TRUE;

        -- All other S-record types are ignored
        when others =>
          assert FALSE report "Unrecognized S-record (eprom.s19)"
            severity warning;
      end case;
      assert (done or not endfile(rom_file)) report 
        "Premature end of S-record file (no S9 record)" severity note;
    end loop;

    -- Report stastics
    write(str,total);
    write(str," bytes read in ");
    write(str,n_recs);
    write(str," S-records.");
    writeline(output,str);

    bus_data <= "ZZZZZZZZ";

    -- loop for every bus cycle taking read and write requests
    loop
      -- wait until address strobe and address in register block range
      wait until as='0';
      while reset='0' loop
        wait until as='0';
      end loop;

      -- Latch address and rw at address strobe
      address:=to_integer(unsigned(bus_addr));
      rw:=bus_rw;

      wait until ph1='1';  -- wait for start data valid

      if ((address>=base) and (address<(base+size))) then
        address:=address-base;  -- adjust for indexing
        if (rw='1') then -- read 
          bus_data<=mem(address);
          wait until ph1='0';
          bus_data<="ZZZZZZZZ";  -- remove driver at end of data valid
        else -- write
          mem(address):=bus_data;
          wait until ph1='0';
        end if;
      end if;
    end loop;
  end process;
end behavior;
