--
-- GM HC11 CPU Core
-- Copyright (C) Green Mountain Computing Systems, 2000
-- All rights reserved.
--
-- This file may not be freely distributed.  This file has been provided
-- under the terms of the GM Core License Agreement in license.txt.
--
-- ram.vhd : This is the VHDL behavioral implementation of the HC11's
--        small on-chip RAM.
--
-- 8/22/00 : Created - Scott Thibault
--

library ieee;
use ieee.std_logic_1164.all;
use ieee.numeric_std.all;

entity hc11ram is
  generic (size : natural :=512);    -- Size of RAM in bytes
  port (E : in std_logic;
        ph1, ph2, reset : in std_logic;

        -- System bus signals
        as, bus_rw : in std_logic;
        bus_addr : in std_logic_vector(15 downto 0);
        bus_data : inout std_logic_vector(7 downto 0));
end hc11ram;

use textio.all;

architecture behavior of hc11ram is
begin
  ram: process
    -- variables for reads and writes
    type mem_array is array (0 to size-1) of std_logic_vector(7 downto 0);  -- An array type 
    variable mem : mem_array;   -- The contents of the RAM
    variable rw : std_logic;          -- Latched bus rw value
    variable address: integer;  -- Latched bus address value
  begin
    bus_data<="ZZZZZZZZ";
    mem(0):="01111110";
    mem(1):="11111000";
    mem(2):="00000000";

    -- loop for every taking read and write requests
    loop
      -- wait until address strobe and address in register block range
      wait until as='0';
      while reset='0' loop
        wait until as='0';
      end loop;

      -- Latch address and rw at address strobe
      address:=to_integer(unsigned(bus_addr));
      rw:=bus_rw;

      wait until ph1='1';  -- wait for start data valid

      if (address<size) then
        if (rw='1') then -- read 
          bus_data<=mem(address);
          wait until ph1='0';
          bus_data<="ZZZZZZZZ";  -- remove driver at end of data valid
        else -- write
          mem(address):=bus_data;
          wait until ph1='0';
        end if;
      end if;
    end loop;
  end process;
end behavior;
